<?php
// XAMPP-Lite
// https://xampplite.sf.net

include_once __DIR__ . '/Mysqldump.php';

// Constants
define('MYSQL_PORT', getenv('MYSQL_PORT') ?: '3306');
define('EXPORT_DIR', __DIR__ . "/../../tmp/db_exports");
define('LOG_FILE', EXPORT_DIR . DIRECTORY_SEPARATOR . "_errors.log");

// Create MySQLi connection
function getMysqli() {
    return new mysqli('127.0.0.1', 'root', '', '', (int) MYSQL_PORT);
}

// Logging function
function logError($message) {
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[$timestamp] $message" . PHP_EOL;
    file_put_contents(LOG_FILE, $logMessage, FILE_APPEND | LOCK_EX);
}

// Fetch database list
function getDatabaseList($mysqli) {
    $dbs = [];
    if ($mysqli->connect_errno) {
        logError("Cannot connect to MySQL/MariaDB: " . $mysqli->connect_error);
        return [];
    }

    $result = $mysqli->query('SHOW DATABASES WHERE `Database` NOT IN '
        . '("information_schema", "phpmyadmin", "mysql", "performance_schema", "sys")');
    
    while ($row = $result->fetch_assoc()) {
        $dbs[] = $row['Database'];
    }
    
    return $dbs;
}

// Refresh the page
function refreshPage() {
    header("Location: ./", true, 303);
    exit;
}

// Write SQL headers to file
function writeSqlHeaders($fileHandle, $versionInfo) {
    $headers = [
        "-- XAMPP-Lite " . PHP_VERSION,
        "-- https://xampplite.sf.net/",
        "-- ----------------------------------------------------------------",
        "-- Server version: " . $versionInfo,
        "-- Date: " . date('r'),
        "",
        "-- MySQL compatibility and import directives (DO NOT REMOVE)",
        "/*!40101 SET NAMES utf8 */;",
        "/*!50503 SET NAMES utf8mb4 */;",
        "/*!40101 SET @OLD_CHARACTER_SET_CLIENT=@@CHARACTER_SET_CLIENT */;",
        "/*!40101 SET @OLD_CHARACTER_SET_RESULTS=@@CHARACTER_SET_RESULTS */;",
        "/*!40103 SET @OLD_TIME_ZONE=@@TIME_ZONE */;",
        "/*!40103 SET TIME_ZONE='+00:00' */;",
        "/*!40014 SET @OLD_FOREIGN_KEY_CHECKS=@@FOREIGN_KEY_CHECKS, FOREIGN_KEY_CHECKS=0 */;",
        "/*!40101 SET @OLD_SQL_MODE=@@SQL_MODE, SQL_MODE='NO_AUTO_VALUE_ON_ZERO' */;",
        "/*!40014 SET @OLD_UNIQUE_CHECKS=@@UNIQUE_CHECKS, UNIQUE_CHECKS=0 */;",
        "/*!40111 SET @OLD_SQL_NOTES=@@SQL_NOTES, SQL_NOTES=0 */;",
        "",
        ""
    ];
    
    fwrite($fileHandle, implode(PHP_EOL, $headers));
}

// Write SQL footers to file
function writeSqlFooters($fileHandle) {
    $footers = [
        "/*!40101 SET CHARACTER_SET_CLIENT=@OLD_CHARACTER_SET_CLIENT */;",
        "/*!40101 SET CHARACTER_SET_RESULTS=@OLD_CHARACTER_SET_RESULTS */;",
        "/*!40103 SET TIME_ZONE=IFNULL(@OLD_TIME_ZONE, 'system') */;",
        "/*!40014 SET FOREIGN_KEY_CHECKS=@OLD_FOREIGN_KEY_CHECKS */;",
        "/*!40101 SET SQL_MODE=@OLD_SQL_MODE */;",
        "/*!40014 SET UNIQUE_CHECKS=@OLD_UNIQUE_CHECKS */;",
        "/*!40111 SET SQL_NOTES=IFNULL(@OLD_SQL_NOTES, 1) */;",
        "",
        ""
    ];
    
    fwrite($fileHandle, implode(PHP_EOL, $footers));
}

// Export a single database
function exportDatabase($db, $versionInfo) {
    try {
		if (!is_dir(EXPORT_DIR)) {
			mkdir(EXPORT_DIR, 0777, true);
		}
        $date = date('Y_m_d');
        $filename = EXPORT_DIR . DIRECTORY_SEPARATOR . "databases_{$date}.sql";
        $fileHandle = fopen($filename, "w");
        
        $dumpOptions = [
            'add-locks' => false,
            'complete-insert' => true,
            'disable-keys' => false,
            'no-autocommit' => false,
            'lock-tables' => false
        ];

        $dsn = (MYSQL_PORT == '3306')
            ? "mysql:host=127.0.0.1;charset=utf8mb4"
            : "mysql:host=127.0.0.1:" . MYSQL_PORT . ";charset=utf8mb4";

        writeSqlHeaders($fileHandle, $versionInfo);
        
        $tempFilename = tempnam(EXPORT_DIR, 'db_dump_');
        $currentDsn = "$dsn;dbname=$db";
        $dump = new Mysqldump($currentDsn, 'root', '', $dumpOptions);
        $dump->start($tempFilename);
        
        fwrite($fileHandle, file_get_contents($tempFilename));
        unlink($tempFilename);
        
        writeSqlFooters($fileHandle);
        fwrite($fileHandle, "-- Dump completed on: " . date('r') . PHP_EOL);

        fclose($fileHandle);

        // Output the file
        header("Content-Disposition: attachment; filename=\"{$db}.sql\"");
        header('Content-Type: application/octet-stream');
        header("Content-Length: " . filesize($filename));
        ob_clean();
        flush();
        readfile($filename);
        exit;
    } catch (Exception $e) {
        logError("Database export failed [{$db}]: " . $e->getMessage());
        refreshPage();
    }
}

// Drop a database
function dropDatabase($db, $mysqli) {
    try {
        if (!$mysqli->query("DROP DATABASE `" . $mysqli->real_escape_string($db) . "`")) {
            logError("Database drop failed [{$db}]: " . $mysqli->error);
        }
    } catch (Exception $e) {
        logError("Database drop exception [{$db}]: " . $e->getMessage());
        refreshPage();
    }
}

// Export all databases
function exportAllDatabases($dbs, $versionInfo) {
    if (empty($dbs)) {
        refreshPage();
    }

    try {
		if (!is_dir(EXPORT_DIR)) {
			mkdir(EXPORT_DIR, 0777, true);
		}
        $date = date('Y_m_d');
        $filename = EXPORT_DIR . DIRECTORY_SEPARATOR . "databases_{$date}.sql";
        $fileHandle = fopen($filename, "w");
        
        $dumpOptions = [
            'add-locks' => false,
            'complete-insert' => true,
            'disable-keys' => false,
            'no-autocommit' => false,
            'lock-tables' => false,
            'databases' => true,
            'add-drop-database' => true
        ];

        $dsn = (MYSQL_PORT == '3306')
            ? "mysql:host=127.0.0.1;charset=utf8mb4"
            : "mysql:host=127.0.0.1:" . MYSQL_PORT . ";charset=utf8mb4";

        writeSqlHeaders($fileHandle, $versionInfo);

        foreach ($dbs as $i => $db) {
            try {
                $tempFilename = tempnam(EXPORT_DIR, 'db_dump_');
                $currentDsn = "$dsn;dbname=$db";
                $dump = new Mysqldump($currentDsn, 'root', '', $dumpOptions);
                $dump->start($tempFilename);

                $header = "-- " . str_repeat("■", 64) . PHP_EOL
                    . "-- " . str_pad(" Database " . ($i + 1) . " of " . count($dbs) . ": `$db` ", 64, "=", STR_PAD_BOTH) . PHP_EOL
                    . "-- " . str_repeat("■", 64) . PHP_EOL;

                fwrite($fileHandle, $header);
                fwrite($fileHandle, file_get_contents($tempFilename));
                fwrite($fileHandle, PHP_EOL);
                unlink($tempFilename);
            } catch (Exception $e) {
                logError("Export failed for database [{$db}]: " . $e->getMessage());
                refreshPage();
            }
        }
        
        writeSqlFooters($fileHandle);
        $footer = "-- " . str_repeat("■", 64) . PHP_EOL
            . "-- Dump completed on: " . date('r') . PHP_EOL
            . "-- " . str_repeat("■", 64) . PHP_EOL;

        fwrite($fileHandle, $footer);
        fclose($fileHandle);

        // Output the file
        header("Content-Disposition: attachment; filename=\"databases_{$date}.sql\"");
        header('Content-Type: application/octet-stream');
        header("Content-Length: " . filesize($filename));
        ob_clean();
        flush();
        readfile($filename);
    } catch (Exception $e) {
        logError("Bulk export failed: " . $e->getMessage());
        refreshPage();
    }
}

// Main
try {
    $mysqli = getMysqli();
    $dbs = getDatabaseList($mysqli);

    if (isset($_REQUEST['exportdb'])) {
        $db = htmlspecialchars($_REQUEST['exportdb']);
        if (in_array($db, $dbs)) {
            exportDatabase($db, $mysqli->server_info);
        }
        refreshPage();
    } elseif (isset($_REQUEST['dropdb'])) {
        $db = htmlspecialchars($_REQUEST['dropdb']);
        if (in_array($db, $dbs)) {
            dropDatabase($db, $mysqli);
        }
        refreshPage();
    } elseif (isset($_REQUEST['exportalldbs'])) {
        exportAllDatabases($dbs, $mysqli->server_info);
    }
} catch (Exception $e) {
    logError("Critical error: " . $e->getMessage());
    refreshPage();
}

// Find Latest Adminer version
function findLatestAdminerVersion() {
  $latestVersion = null;
  foreach (glob(__DIR__ . DIRECTORY_SEPARATOR . 'adminer' . DIRECTORY_SEPARATOR . 'adminer-*.php') as $file) {
      if (preg_match('/adminer-(\d+\.\d+\.\d+)/', $file, $matches)) {
          $version = $matches[1];
          if (is_null($latestVersion) || version_compare($version, $latestVersion) > 0) {
              $latestVersion = $version;
          }
      }
  }
  return $latestVersion;
}

?>
<!doctype html>
<html lang="en">

<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <link rel="icon"
    href="data:image/x-icon;base64,iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAACXBIWXMAABDfAAAQ3wF0VV4wAAAAGXRFWHRTb2Z0d2FyZQB3d3cuaW5rc2NhcGUub3Jnm+48GgAABV1JREFUWIXFl3tsU3UUxz+/373tutJCyzYe614QMDzEwDKYxAgRhflkghoSIMGEYCAkKIhCiKLE8DAEiBEjiS/APwCJ4msQIREwmiAbD6MSmYSNlU1grBvrNtb27v78o9CuvXcbGiMnadLzO4/v95x7f+e0IrpmxFIBGwEv/6+ElWC1iK0Z0XoXwBMk5F0EB/DKuwgOgN6nhy+AKCxGON2olnpUbRXEOu19HS5EUQnCl4eKtqMunYaW+n9JwJONVr4OMWZG6nlHM+aRbZgn96Qcy9J5yEdeArcveagU6twRur5aC+1NtjAitmaEsgVfvB/hz+uRn3nsfcwjW+PgZa8ipyzq0VeFgnTteM6WhO07IGeu6xUcQE5djCgqQQy/v1dwADEwHznzTVub9RH4Asgx0xNqR/AcFz9+mUhjHVml5RTOewshNRACNfl5kMkUqsugZtcqmk4cwJkVYOSSHbgLxsYJjy3D9OVCS0NqIen4qqAYhEjowZ0rGdxazUh3BCr30HTiQLKywhJEYXFCv3bsU/RfvmC0X5IfayC4c2W3NghU/oS+O9ApM3F00wdHG/C44jzdDp0bobqETfNmJYmbBp3HPyLgiad06QJ581Jqbq0fmWl4lg7Emi6n6J6SmUlFc9C/+HFLFQChym8ZGLmScuYdNy0td9ASZ+mAqqnEaGtG9/jjDk+9gcofj2oOIsaWIYaMsiQxozdp+XozhbpIzTV5QeK70RaC2krQ+iDg0WLUf7mZwvkb4gdSQxTPRqT5RRrriDZfIRqqp/GHPQSMq9CNgMi7D23YxIR++fO3GSqNvjugCRCVe2jwB8h9YmlapZ1c/X4n147uJnr9MroUODXIdmtkZnYrTQiYcesFVIqGiu1opz9D81gvne0gUkBtSwyjYCJDpi/EM6IEoTuo3rYAvf4sOf00PE5p6cptkaVzUQ8to/XPSq4c/gA9eIoin8PW33YUC2CYz0Grx4mmSYQQ1Hy4nJzQr/j9DruQZOyEWcgn12K0t6BpkiFeJ15fzzH2oxiQZa8gp7wAwI0z33Fz9xKy3ZqdazxR/njkg4sQY2dYbObxHZiHt9jG2XZAls5NgAM0HXyHgjRwWTwbMfph8GQjBhaAJ7tHcnLqYmiux6zcewcE9AzEI8sTarj6Z/wt58GZHBkiexjy6fWgWcM7gucInTqEK6eArMmz42MbYPoKOHMAjEgfBIZNQnRbqc1HPyHXmTavpi2zB6/7neqN5Xg1g1ZD0VZzlqL56+Nd6OfHLCqBCz+lxFgmYdQ7NKkoRcbFH1Md/AHkuEdv2c3455a0njnEKB8UDnBwT5YT/bdvUnP3D1hIWwl0tCUVIfAPGppiFw8sRNzagLFwCKO9JWHzDilCl8nLNihveGru9jbSxULAqD0NKnkx9Gc3xX8baDqyeDZa6dyErf1CFW0XqhJ6v4mzkKVzweVFBO5Ff2ZTMrFSGLVJ30RB6dew01DE5ryLv/ixVE+zC2S3m6AUdVvngBAUrNhnSZwuoaoKnPtfxJW2LywdcOmC63tfI9JYl2qQqdewoWI77oZTZNZX0VDxXq/gkcY6ru973QJuSwBgqLjBHxvKCVUdTHkcEN9qNbtW0XFoCz6XxO+SdBzcTO3u1fGN112UIlRVwfmN5QS0VltyPU7CTkPF90H/XLwjS5AZbiLXg7RVnyQno4tcr56Y7QqoDxs0RXU8IyfhzM7D7GwnfKEKvfUvinwO2+p7JXBbwlGTtqiJqcCpCQZkSJyafbJol+JGxCTapZACPE6JN32GpEmff0y8d5Dktjg1QU4v+8JOJBD+RxH/rYSlEqy+SyTCSrHqbw9xyy0dDHQ9AAAAAElFTkSuQmCC">
  <title>XAMPP-Lite Admin</title>
  <meta name="description"
    content="xampp-lite is an easy to install Apache distribution containing MariaDB, PHP, phpMyAdmin, HeidiSql, Adminer and other tools" />
  <meta name="keywords" content="xampp-lite, apache, php, mariadb, open source distribution" />
  <link rel="stylesheet" href="style.css" />
  <script>
    window.onload = function () {
      document.getElementById("loader-wrapper").style.display = "none";
    };
  </script>
</head>

<body class="index">
    <noscript>
        <div class="noscript-message">
            JavaScript is disabled in your browser. Some functionalities may not work properly.
        </div>
    </noscript>
  <!--[if lte IE 11]>
    <div class="warning-ie8">
        <p>Your browser is outdated and does not support PhpMyAdmin. Please upgrade to a recent browser.</p>
        <a href='/adminer' target='_blank' title="Adminer: Lightweight Database Management Tool"
        alt="Adminer">Launch Adminer</a>
    </div>
<![endif]-->
  <div id="loader-wrapper">
    <div id="loader"></div>
  </div>
  <div class="row" id="container">
    <div id="webadmin">
      <form method="get" action="/phpmyadmin/index.php" style="display:inline;" target='_blank'>
        <input type="hidden" name="route" value="/server/databases">
        <a href='/phpmyadmin' target='_blank' title="phpMyAdmin: Database Management Interface" alt="phpMyAdmin"
          onclick="this.closest('form').submit();return false;">
          <figure class='admin-button'>
            <svg width="170" height="65" viewBox="0 0 170 65">
              <g style="fill:#f89c0e">
                <g style="fill:#ccc;stroke-width:1.42975">
                  <path
                    d="m2889.39 6.348-2.04-4.07c-1.01-1.01-2.03-2.04-4.06-2.04l-4.08 1.03c-2.03 2.03-3.05 3.05-3.05 5.08l-86.52 1566.347 13.24-2.035 83.46-1523.702 99.75 163.88h-1.02c75.32 221.88 106.87 458.02 94.66 708.41l6.11 9.16 98.73 175.06 6.1 6.11c151.66 133.336 321.63 222.907 509.94 268.707l45.8 74.309 6.11 2.031 1.02-2.031L2887.35 35.768l2.04-29.52"
                    style="stroke-width:1.42975" transform="matrix(.03295 0 0 .0283 41.836 1.117)" />
                  <path
                    d="M2858.86 559.028c106.87 218.84 145.54 549.63 117.05 992.392l416.29-50.894zm948.62 961.853c42.74-5.086 70.22-14.246 82.44-27.476l-2178.16 265.656h-1.02c1.02 90.586 41.73 161.836 121.12 212.723 21.37 15.269 43.77 26.465 64.12 33.59 19.34-22.387 40.72-38.676 66.17-53.946h1.01c228-138.422 566.94-159.8 1014.78-65.136l5.09 1.011c48.86 10.18 97.71 22.399 143.52 36.645 13.22 2.035 24.42-2.035 33.58-10.18 16.29-12.215 36.65-21.375 64.13-27.476v-1.02c72.27-128.25 169.97-222.906 292.12-284.996 84.48-41.727 182.19-69.215 291.1-79.395m-1046.33 39.704 99.74-1528.787c-144.53 557.77-376.6 1078.9-695.18 1562.369l595.44-33.582M2200.32 329.008l-4.07-2.04-4.07 1.02-2.04 5.08-82.44 1323.188 12.21-1.019 82.45-1322.169-2.04-4.06m-116.03 1298.748 87.53-1290.608-461.08 1330.308 373.55-39.7"
                    style="stroke-width:1.42975" transform="matrix(.03295 0 0 .0283 41.836 1.117)" />
                </g>
                <g style="stroke-width:1.14916">
                  <path
                    d="M2102.61 2057.953c-115.02 54.961-198.48 64.121-251.41 30.54-52.93-35.626-134.36-27.486-244.28 24.421m765.41-46.821c-21.37-7.125-44.78-12.211-69.21-14.25-49.87-4.066-109.93 15.27-180.16 57.004-70.23 42.746-154.71 59.035-255.48 49.871m1132.86-196.445c-484.5-99.742-833.61-76.336-1047.35 71.25m1352.7-23.406c-21.38-8.145-44.79-13.235-69.22-14.254-49.87-4.067-109.92 15.269-180.15 57.004-70.23 41.726-154.71 58.011-255.48 49.87"
                    style="fill:none;stroke:#ccc;stroke-width:13.7899;stroke-linecap:round;stroke-linejoin:round"
                    transform="matrix(.03814 0 0 .03784 13.08 -17.44)" />
                </g>
                <g style="stroke-width:1.20633">
                  <path
                    d="M2978.872 853.556c-49.88-30.54-101.78-22.39-155.73 25.44-46.82-47.83-95.67-55.98-146.57-25.44m194.713-144.035c-50.89-30.53-100.76-18.32-151.66 36.64-49.88-54.96-100.76-67.17-151.66-36.64"
                    style="fill:none;stroke:#ccc;stroke-width:34.9835;stroke-linecap:round;stroke-linejoin:round"
                    transform="matrix(.03912 0 0 .03348 14.818 -20.672)" />
                </g>
                <path
                  d="M4.342 81.547H14.75c3.126 0 5.422.888 6.888 2.664 1.464 1.776 1.94 4.25 1.425 7.424a12.461 12.461 0 0 1-1.129 3.585 11.541 11.541 0 0 1-2.147 2.982c-1.058 1.082-2.243 1.858-3.554 2.326-1.312.468-2.99.702-5.032.702H6.565l-1.153 7.118H0Zm4.725 4.24L7.246 97.03h3.291c2.182 0 3.8-.444 4.852-1.334 1.052-.89 1.746-2.37 2.082-4.443.323-1.998.135-3.408-.564-4.23-.699-.823-2.076-1.235-4.132-1.235H9.067m20.071-11.359h5.371l-1.153 7.118h4.778c3.008 0 5.092.569 6.254 1.705 1.162 1.137 1.539 2.969 1.13 5.496l-2.023 12.482h-5.453l1.925-11.88c.22-1.35.113-2.268-.318-2.755-.431-.486-1.333-.73-2.705-.73h-4.287l-2.49 15.365h-5.371l4.342-26.801m22.229 7.118h10.406c3.128 0 5.423.888 6.888 2.664 1.465 1.776 1.94 4.25 1.426 7.424a12.48 12.48 0 0 1-1.128 3.585 11.567 11.567 0 0 1-2.148 2.982c-1.059 1.082-2.243 1.858-3.555 2.326-1.31.468-2.988.702-5.031.702H53.59l-1.153 7.118h-5.412zm4.725 4.24L54.271 97.03h3.292c2.181 0 3.798-.444 4.85-1.334 1.052-.89 1.747-2.37 2.082-4.443.324-1.998.136-3.408-.563-4.23-.699-.823-2.076-1.235-4.132-1.235h-3.708"
                  transform="translate(-.009 1.124) scale(.56665)" style="fill:#6c78af" />
                <path
                  d="M44.891 40.072h5.897l2.591 14.12 7.167-14.12h5.876l-2.98 18.402h-3.939L62.15 44.14l-7.227 14.334h-4.266L47.941 43.99 46 58.474h-4.09l2.981-18.402M73.5 55.496h2.978l1.707-10.536h3.701l-2.13 13.15c-.29 1.789-.938 3.113-1.943 3.974-1.006.86-2.4 1.29-4.185 1.29h-7.26l.445-2.737h6.621c.71 0 1.284-.166 1.72-.497.434-.331.703-.811.804-1.44l.037-.227H72.72c-2.096 0-3.525-.397-4.288-1.195-.762-.797-.992-2.126-.692-3.987l1.35-8.331h3.66l-1.305 8.055c-.165 1.02-.115 1.688.15 2.005.266.317.9.476 1.906.476m18.356-15.424h4.262l5.166 18.402h-4.47l-1.046-4.38h-8.302l-2.412 4.38h-4.263Zm1.383 3.654-3.94 7.083h5.69l-1.75-7.083m22 14.748h-7.11c-2.15 0-3.728-.61-4.735-1.829-1.008-1.219-1.333-2.918-.98-5.097a8.52 8.52 0 0 1 .775-2.461 7.93 7.93 0 0 1 1.475-2.048c.727-.744 1.544-1.275 2.45-1.597.905-.322 2.056-.482 3.45-.482h3.176l.792-4.888h3.688zm-3.219-2.898 1.25-7.705h-2.239c-1.502 0-2.616.303-3.344.908-.726.606-1.205 1.614-1.433 3.026-.222 1.37-.09 2.34.393 2.913.485.572 1.43.858 2.836.858h2.538m8.625-10.616h14.215c2.061 0 3.484.388 4.267 1.164.783.776 1.034 2.036.752 3.78l-1.389 8.57h-3.73l1.313-8.107c.162-.994.112-1.649-.147-1.966-.26-.318-.875-.476-1.844-.476h-2.077l-1.709 10.549h-3.785l1.71-10.55h-4.327l-1.71 10.55h-3.728l2.19-13.515m27.474-1.535h-3.882l.543-3.352h3.881zm-2.439 15.05h-3.88l2.19-13.515h3.88l-2.189 13.515m5.391-13.514h6.965c2.105 0 3.547.382 4.324 1.144.776.765 1.022 2.03.735 3.8l-1.388 8.57h-3.702l1.318-8.132c.164-1.01.107-1.666-.17-1.966-.276-.301-.904-.452-1.882-.452h-2.978l-1.71 10.55h-3.7l2.188-13.515" />
              </g>
            </svg>

          </figure>
        </a>
      </form>
      <a href='/adminer' target='_blank' title="Adminer: Lightweight Database Management Tool" alt="Adminer">
        <figure class='admin-button'>
          <svg width="170" height="65">
            <g stroke="#1c274c" stroke-width=".106" style="fill:none;fill-opacity:.518939;stroke:#00567e;stroke-opacity:1">
                <path stroke-linecap="round" d="M.158 1.51V.659m1.135 0v.85" style="fill:none;fill-opacity:.518939;stroke:#00567e;stroke-opacity:1" transform="matrix(37.57779 0 0 37.57779 -3.899 -7.964)"/>
                <path d="M.726.942c.313 0 .567-.127.567-.283 0-.157-.254-.284-.567-.284-.314 0-.568.127-.568.284 0 .156.254.283.568.283zm.567.142c0 .157-.254.284-.567.284-.314 0-.568-.127-.568-.284m1.135.426c0 .156-.254.283-.567.283-.314 0-.568-.127-.568-.283" style="fill:none;fill-opacity:.518939;stroke:#00567e;stroke-opacity:1" transform="matrix(37.57779 0 0 37.57779 -3.899 -7.964)"/>
            </g>
            <text xml:space="preserve" x="50.05" y="57.675" style="font-size:33.3423px;line-height:2;fill:#00567e;fill-opacity:.518939;stroke:#00567e;stroke-width:1.1114;stroke-dasharray:none;stroke-opacity:1">
                <tspan x="50.05" y="57.675" style="font-style:normal;font-variant:normal;font-weight:700;font-stretch:normal;font-size:33.3423px;font-family:Calibri;-inkscape-font-specification:&quot;Calibri Bold&quot;;fill:#00567e;fill-opacity:.518939;stroke:#00567e;stroke-width:1.1114;stroke-dasharray:none;stroke-opacity:1">Adminer</tspan>
            </text>
            <text xml:space="preserve" x="51.448" y="22.436" style="font-weight:700;font-size:24.5379px;font-family:&quot;Comic Sans MS&quot;;-inkscape-font-specification:&quot;Comic Sans MS Bold&quot;;fill:red;fill-opacity:1;stroke:none;stroke-width:1.60825;stroke-linejoin:round;stroke-dasharray:9.28246,1.54707;stroke-opacity:1">
                <tspan x="51.448" y="22.436" style="font-style:normal;font-variant:normal;font-weight:700;font-stretch:normal;font-size:24.5379px;font-family:Consolas;-inkscape-font-specification:&quot;Consolas Bold&quot;;fill:red;fill-opacity:1;stroke:none;stroke-width:1.60825;stroke-opacity:1">&lt;?</tspan>
            </text>
            <text xml:space="preserve" x="90.197" y="29.15" style="font-style:normal;font-variant:normal;font-weight:700;font-stretch:normal;font-size:17.7219px;font-family:Consolas;-inkscape-font-specification:&quot;Consolas Bold&quot;;fill:#00567e;fill-opacity:1;stroke:none;stroke-width:1.60825;stroke-linejoin:round;stroke-dasharray:9.28246,1.54707;stroke-opacity:1">
                <tspan x="90.197" y="29.15" style="font-style:normal;font-variant:normal;font-weight:700;font-stretch:normal;font-size:17.7219px;font-family:Consolas;-inkscape-font-specification:&quot;Consolas Bold&quot;;fill:#00567e;fill-opacity:1;stroke-width:1.60825">v<?php echo findLatestAdminerVersion()?></tspan>
            </text>
          </svg>
        </figure>
      </a>
    </div>
    <div id="exportbd">
      <form id="db-form" action="#" method="get">
        <?php
          if(!isset($dbs) || count($dbs) === 0){
            echo "<table><tr><th colspan='2' style='text-align:center'>Databases</th></tr>";
          } else {
            echo "<table><tr><th>Databases</th><th><button name='exportalldbs' class='export-all-button'>Export all Databases</button></th></tr>";
            foreach ($dbs as $db) {
              echo "<tr>
                      <td class='to-right'>{$db}</td>
                      <td>
                        <button name='exportdb' value='{$db}' class='export-button'>Export</button>
                        <button type='button' value='{$db}' class='drop-button' onclick='showConfirmationModal(this.value)'>Drop</button>
                      </td>
                    <tr>";
            }
            echo "</table>";
        }
        ?>
      </form>
    </div>
  </div>
  <div id='xampp-lite-home'><a href='https://xampplite.sf.net/' target='_blank'
      title="Visit XAMPP-Lite Website"><span>https://</span>xampplite<span>.sf.net/</span></a></div>
<?php  
  $php_man = getenv('XAMPP_LITE_ROOT') . '/apps/php/phpmanual/langref.html';
  if (is_file($php_man)) {
?>
  <a href='/phpmanual/langref.html' target='_blank' title="PHP Language Reference">
    <figure id='phpmanual'><svg width="32" height="32" viewBox="0 0 32 32">
        <rect width="30.897" height="32" x="-32" y="-32" rx="1.545" ry="1.6" transform="matrix(0 -1 -1 0 0 0)"
          style="opacity:.2;stroke-width:1.12297" />
        <rect style="fill:#465489;stroke-width:1.12297" width="30.897" height="32" x="-30.897" y="-32" rx="1.545"
          ry="1.6" transform="matrix(0 -1 -1 0 0 0)" />
        <path
          d="m14.592 11.941-1.352 6.941h1.43l.733-3.762c.486 0 .948-.016 1.408.004.452.02.623.239.547.683-.105.607-.228 1.211-.343 1.816l-.245 1.272c.468 0 .916.001 1.364-.006.037 0 .095-.058.106-.1.04-.155.064-.316.094-.474.172-.899.356-1.797.514-2.698.14-.795-.189-1.3-.966-1.465-.369-.079-.753-.089-1.131-.106-.374-.016-.75-.002-1.138-.002l.408-2.103h-1.429Zm-6.374 2.103c-.044 0-.089.006-.136.01-.428 2.201-.993 4.742-1.421 6.943h1.441c.109-.553.357-1.437.455-1.975.02-.104.06-.13.158-.129.473.002.947.013 1.419-.008.297-.013.598-.052.887-.123 1.083-.265 2.18-1.42 2.058-3.057-.063-.84-.538-1.359-1.358-1.559a3.43 3.43 0 0 0-.768-.098c-.912-.011-1.823-.004-2.735-.004zm11.873 0c-.034 0-.067.005-.108.01-.428 2.197-.994 4.742-1.422 6.946h1.44c.107-.549.356-1.434.454-1.971.02-.113.067-.137.17-.137.485.003.97.01 1.454-.006.64-.021 1.243-.166 1.758-.578.795-.636 1.162-1.474 1.153-2.476-.008-.853-.5-1.468-1.328-1.678a3.416 3.416 0 0 0-.785-.105c-.928-.012-1.857-.005-2.786-.005zM9.33 15.054c.544.032 1.082.032 1.609.104.461.064.657.33.637.797a3.679 3.679 0 0 1-.142.845c-.174.608-.605.945-1.224 1.009-.462.048-.93.042-1.428.062l.548-2.817zm11.9.002c.543.031 1.08.03 1.608.102.462.063.659.327.638.794a3.668 3.668 0 0 1-.152.878c-.178.59-.608.917-1.211.979-.462.048-.928.042-1.43.062l.547-2.816z"
          style="opacity:.2;isolation:isolate;fill-rule:evenodd" transform="matrix(1.14286 0 0 1.10345 -2.286 -6.62)" />
        <path style="isolation:isolate;fill:#fff;fill-rule:evenodd"
          d="m14.592 10.941-1.352 6.941h1.43l.733-3.762c.486 0 .948-.016 1.408.004.452.02.623.239.547.683-.105.607-.228 1.211-.343 1.816l-.245 1.272c.468 0 .916.001 1.364-.006.037 0 .095-.058.106-.1.04-.155.064-.316.094-.474.172-.899.356-1.797.514-2.698.14-.795-.189-1.3-.966-1.465-.369-.079-.753-.089-1.131-.106-.374-.016-.75-.002-1.138-.002l.408-2.103h-1.429Zm-6.374 2.103c-.044 0-.089.006-.136.01-.428 2.201-.993 4.742-1.421 6.943h1.441c.109-.553.357-1.437.455-1.975.02-.104.06-.13.158-.129.473.002.947.013 1.419-.008.297-.013.598-.052.887-.123 1.083-.265 2.18-1.42 2.058-3.057-.063-.84-.538-1.359-1.358-1.559a3.43 3.43 0 0 0-.768-.098c-.912-.011-1.823-.004-2.735-.004zm11.873 0c-.034 0-.067.005-.108.01-.428 2.197-.994 4.742-1.422 6.946h1.44c.107-.549.356-1.434.454-1.971.02-.113.067-.137.17-.137.485.003.97.01 1.454-.006.64-.021 1.243-.166 1.758-.578.795-.636 1.162-1.474 1.153-2.476-.008-.853-.5-1.468-1.328-1.678a3.416 3.416 0 0 0-.785-.105c-.928-.012-1.857-.005-2.786-.005zM9.33 14.054c.544.032 1.082.032 1.609.104.461.064.657.33.637.797a3.679 3.679 0 0 1-.142.845c-.174.608-.605.945-1.224 1.009-.462.048-.93.042-1.428.062l.548-2.817zm11.9.002c.543.031 1.08.03 1.608.102.462.063.659.327.638.794a3.668 3.668 0 0 1-.152.878c-.178.59-.608.917-1.211.979-.462.048-.928.042-1.43.062l.547-2.816z"
          transform="matrix(1.14286 0 0 1.10345 -2.286 -6.62)" />
        <path style="opacity:.1;fill:#fff;stroke-width:1.12297"
          d="M1.6 0C.714 0 0 .69 0 1.545V2.65c0-.856.714-1.546 1.6-1.546h28.8c.886 0 1.6.69 1.6 1.546V1.545C32 .69 31.286 0 30.4 0Z" />
        <path
          d="M442 124c0 1.104-.927 1.656-2 2 0 0-5.443 1.515-11 2.977V104l11-3a2 2 0 0 1 2 2zm-15 4.998c-5.462-1.468-11-2.998-11-2.998-1.136-.312-2-.896-2-2v-21a2 2 0 0 1 2-2l11 3zM440 99s-5.789 1.594-11.05 3c-.659.025-1.323 0-1.983 0-5.012-1.344-10.967-3-10.967-3a4 4 0 0 0-4 4v21c0 2.209 1.885 3.313 4 4 0 0 5.393 1.5 10.967 3h2.025c5.62-1.5 11.008-3 11.008-3 2.053-.531 4-1.791 4-4v-21a4 4 0 0 0-4-4z"
          style="fill:#fff;stroke:#fdfdfd;stroke-width:1.20783;stroke-dasharray:none;stroke-opacity:1"
          transform="matrix(.4199 0 0 .32963 -163.716 -15.47)" fill="#000" fill-rule="evenodd" />
      </svg>
    </figure>
  </a>
<?php } ?>
  <a href='/phpinfo' target='_blank' title="PHP Configuration">
    <figure id='phpinfo'><svg width="32" height="32" viewBox="0 0 32 32">
        <rect width="30.897" height="32" x="-32" y="-32" rx="1.545" ry="1.6" transform="matrix(0 -1 -1 0 0 0)"
          style="opacity:.2;stroke-width:1.12297" />
        <rect style="fill:#465489;stroke-width:1.12297" width="30.897" height="32" x="-30.897" y="-32" rx="1.545"
          ry="1.6" transform="matrix(0 -1 -1 0 0 0)" />
        <path
          d="m14.592 11.941-1.352 6.941h1.43l.733-3.762c.486 0 .948-.016 1.408.004.452.02.623.239.547.683-.105.607-.228 1.211-.343 1.816l-.245 1.272c.468 0 .916.001 1.364-.006.037 0 .095-.058.106-.1.04-.155.064-.316.094-.474.172-.899.356-1.797.514-2.698.14-.795-.189-1.3-.966-1.465-.369-.079-.753-.089-1.131-.106-.374-.016-.75-.002-1.138-.002l.408-2.103h-1.429Zm-6.374 2.103c-.044 0-.089.006-.136.01-.428 2.201-.993 4.742-1.421 6.943h1.441c.109-.553.357-1.437.455-1.975.02-.104.06-.13.158-.129.473.002.947.013 1.419-.008.297-.013.598-.052.887-.123 1.083-.265 2.18-1.42 2.058-3.057-.063-.84-.538-1.359-1.358-1.559a3.43 3.43 0 0 0-.768-.098c-.912-.011-1.823-.004-2.735-.004zm11.873 0c-.034 0-.067.005-.108.01-.428 2.197-.994 4.742-1.422 6.946h1.44c.107-.549.356-1.434.454-1.971.02-.113.067-.137.17-.137.485.003.97.01 1.454-.006.64-.021 1.243-.166 1.758-.578.795-.636 1.162-1.474 1.153-2.476-.008-.853-.5-1.468-1.328-1.678a3.416 3.416 0 0 0-.785-.105c-.928-.012-1.857-.005-2.786-.005zM9.33 15.054c.544.032 1.082.032 1.609.104.461.064.657.33.637.797a3.679 3.679 0 0 1-.142.845c-.174.608-.605.945-1.224 1.009-.462.048-.93.042-1.428.062l.548-2.817zm11.9.002c.543.031 1.08.03 1.608.102.462.063.659.327.638.794a3.668 3.668 0 0 1-.152.878c-.178.59-.608.917-1.211.979-.462.048-.928.042-1.43.062l.547-2.816z"
          style="opacity:.2;isolation:isolate;fill-rule:evenodd" transform="matrix(1.14286 0 0 1.10345 -2.286 -6.62)" />
        <path style="isolation:isolate;fill:#fff;fill-rule:evenodd"
          d="m14.592 10.941-1.352 6.941h1.43l.733-3.762c.486 0 .948-.016 1.408.004.452.02.623.239.547.683-.105.607-.228 1.211-.343 1.816l-.245 1.272c.468 0 .916.001 1.364-.006.037 0 .095-.058.106-.1.04-.155.064-.316.094-.474.172-.899.356-1.797.514-2.698.14-.795-.189-1.3-.966-1.465-.369-.079-.753-.089-1.131-.106-.374-.016-.75-.002-1.138-.002l.408-2.103h-1.429Zm-6.374 2.103c-.044 0-.089.006-.136.01-.428 2.201-.993 4.742-1.421 6.943h1.441c.109-.553.357-1.437.455-1.975.02-.104.06-.13.158-.129.473.002.947.013 1.419-.008.297-.013.598-.052.887-.123 1.083-.265 2.18-1.42 2.058-3.057-.063-.84-.538-1.359-1.358-1.559a3.43 3.43 0 0 0-.768-.098c-.912-.011-1.823-.004-2.735-.004zm11.873 0c-.034 0-.067.005-.108.01-.428 2.197-.994 4.742-1.422 6.946h1.44c.107-.549.356-1.434.454-1.971.02-.113.067-.137.17-.137.485.003.97.01 1.454-.006.64-.021 1.243-.166 1.758-.578.795-.636 1.162-1.474 1.153-2.476-.008-.853-.5-1.468-1.328-1.678a3.416 3.416 0 0 0-.785-.105c-.928-.012-1.857-.005-2.786-.005zM9.33 14.054c.544.032 1.082.032 1.609.104.461.064.657.33.637.797a3.679 3.679 0 0 1-.142.845c-.174.608-.605.945-1.224 1.009-.462.048-.93.042-1.428.062l.548-2.817zm11.9.002c.543.031 1.08.03 1.608.102.462.063.659.327.638.794a3.668 3.668 0 0 1-.152.878c-.178.59-.608.917-1.211.979-.462.048-.928.042-1.43.062l.547-2.816z"
          transform="matrix(1.14286 0 0 1.10345 -2.286 -6.62)" />
        <path style="opacity:.1;fill:#fff;stroke-width:1.12297"
          d="M1.6 0C.714 0 0 .69 0 1.545V2.65c0-.856.714-1.546 1.6-1.546h28.8c.886 0 1.6.69 1.6 1.546V1.545C32 .69 31.286 0 30.4 0Z" />
        <path
          d="M1229.93 594.767c36.644 37.975 50.015 91.328 43.72 142.909-9.128 74.877-30.737 144.983-56.093 215.657-27.129 75.623-54.66 151.09-82.332 226.512-44.263 120.685-88.874 241.237-132.65 362.1-10.877 30.018-18.635 62.072-21.732 93.784-3.376 34.532 21.462 51.526 52.648 36.203 24.977-12.278 49.288-28.992 68.845-48.768 31.952-32.31 63.766-64.776 94.805-97.98 15.515-16.605 30.86-33.397 45.912-50.438 11.993-13.583 24.318-34.02 40.779-42.28 31.17-15.642 55.226 22.846 49.582 49.794-5.39 25.773-23.135 48.383-39.462 68.957l-1.123 1.416a1559.53 1559.53 0 0 0-4.43 5.6c-54.87 69.795-115.043 137.088-183.307 193.977-67.103 55.77-141.607 103.216-223.428 133.98-26.65 10.016-53.957 18.253-81.713 24.563-53.585 12.192-112.798 11.283-167.56 3.333-40.151-5.828-76.246-31.44-93.264-68.707-29.544-64.698-8.98-144.595 6.295-210.45 18.712-80.625 46.8-157.388 75.493-234.619l2.18-5.867 1.092-2.934 2.182-5.87 2.182-5.873c33.254-89.517 67.436-178.676 101.727-267.797 31.294-81.296 62.72-162.537 93.69-243.95 2.364-6.216 5.004-12.389 7.669-18.558l1-2.313c6.835-15.806 13.631-31.617 16.176-48.092 6.109-39.537-22.406-74.738-61.985-51.947-68.42 39.4-119.656 97.992-170.437 156.944l-6.175 7.17c-15.78 18.323-31.582 36.607-47.908 54.286-16.089 17.43-35.243 39.04-62.907 19.07-29.521-21.308-20.765-48.637-3.987-71.785 93.18-128.58 205.056-248.86 350.86-316.783 60.932-28.386 146.113-57.285 225.882-58.233 59.802-.707 116.561 14.29 157.774 56.99zm92.038-579.94c76.703 29.846 118.04 96.533 118.032 190.417-.01 169.189-182.758 284.908-335.53 212.455-78.956-37.446-117.358-126.202-98.219-227.002 26.494-139.598 183.78-227.203 315.717-175.87z"
          fill-rule="evenodd" style="fill:#fff" transform="matrix(.0047 0 0 .00455 11.479 18.56)" />
      </svg>
    </figure>
  </a>
  <!-- Confirmation Modal -->
  <div id="confirmationModal" class="modal">
    <div class="modal-content">
      <div class="modal-header">
        <h3 class="modal-title">DROP DATABASE OPERATION</h3>
      </div>
      <div class="modal-body">
        <p>Do you really want to execute:<br>
          "<code id='drop-database-code'>DROP DATABASE `my_database`;</code>"?
        </p>
        <input type="text" 
        id="confirmationInput" 
        class="confirmation-input" 
        placeholder="Type YES to confirm"
        onkeyup="handleEnterKey(event)"
        oninput="hideErrorMessage()">
        <div id="errorMessage" class="error-message">
          Please type "YES" to confirm
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-cancel" onclick="hideConfirmationModal()">Cancel</button>
        <button type="button" id="btn-proceed" class="btn btn-proceed" onclick="validateAndSubmit(this.value)">Proceed</button>
      </div>
    </div>
  </div>
  
  <!-- Script -->
  <script>
        function showConfirmationModal(db_name) {
            const modal = document.getElementById('confirmationModal');
            modal.style.display = 'flex';
            document.getElementById('drop-database-code').innerHTML = `DROP DATABASE \`${db_name}\``;
            document.getElementById('confirmationInput').value = '';
            document.getElementById('btn-proceed').value = db_name;
            document.getElementById('errorMessage').style.display = 'none';
            document.getElementById('confirmationInput').focus();
        }

        function hideConfirmationModal() {
            document.getElementById('confirmationModal').style.display = 'none';
        }
        function hideErrorMessage() {
            document.getElementById('errorMessage').style.display = 'none';
        }
        function validateAndSubmit(db_name) {
            const input = document.getElementById('confirmationInput').value.trim();
            const errorMessage = document.getElementById('errorMessage');
            
            if (input === 'YES') {
                let dbInput = document.createElement("input");
                dbInput.setAttribute("name", "dropdb");
                dbInput.setAttribute("value", db_name);
                dbInput.setAttribute("hidden", "");
                document.getElementById('db-form').appendChild(dbInput);
                document.getElementById('db-form').submit();
            } else {
              if (input.toUpperCase() === 'YES') {
                errorMessage.innerHTML = 'Please type "YES" in CAPITAL letters';
              } else {
                errorMessage.innerHTML = 'Please type "YES" to confirm';
              }
                errorMessage.style.display = 'block';
            }
        }

        function handleEnterKey(event) {
            if (event.key === 'Enter') {
                validateAndSubmit(document.getElementById('btn-proceed').value);
            } else if(event.key === "Escape") {
              hideConfirmationModal();
            }
        }
        // Close modal when clicking outside the modal content
        window.onclick = function(event) {
            const modal = document.getElementById('confirmationModal');
            if (event.target === modal) {
                hideConfirmationModal();
            }
        }
    </script> 
    <noscript>
        <style>
            #loader-wrapper, .drop-button {display:none;}
            /* .drop-button {pointer-events: none;} */
        </style>
    </noscript>    
</body>

</html>